/* ==========================================================================
   File:               xkpasswdApp.js
   Description:	       Application script to use for generating secure passwords.
                       Intended to be used by Keyboard Maestro.
   Copyright:          Apr 2020

   @param {string[]} [process.argv[2]]
   @param {process.argv[2]} [preset]-   (Optional) Number indicating the password
                                        preset to use.
                                        Default:0 (Grumpy Default)
   @param {process.argv[3]} [custom]- ) (Optional) JSON representation of the
                                        HSXKPasswd configuration to use.
                                        Default: Default config.
   ========================================================================== */
'use strict';

const HSXKPasswd = require('hsxkpasswd');

// Enumeration of supported preset types
// for generating passwords.
const XKPASSWD_PRESETS = {
  GRUMPY_DEF: 0,
  DEFAULT   : 1,
  NTLM      : 2,
  SECURITYQ : 3,
  WEB16     : 4,
  WEB32     : 5,
  WIFI      : 6,
  APPLEID   : 7,
  XKCD      : 8,
  CUSTOM    : 9,
};

/* ========================================================================
   Description:    Main function called when the script is invoked.

   @param {stringp[]} [args] - Array of arguments sent to the script.
   @param {args[2]} [preset]-   (Optional) Number indicating the password
                                           preset to use.
                                           Default:0 (Grumpy Default)
   @param {args[3]} [custom]-   (Optional) JSON representation of the
                                           HSXKPasswd configuration to use.
                                          Default: Default config.

   @throws {TypeError}  - thrown when the first 'real' script argument is not
                          boolean-ish.
   @throws {TypeError}  - thrown when the first any of the remaining arguments
                          are not a string.

   @remarks Allows user to specify the hashing algorithm to use.
======================================================================== */
function main(args) {

  if (!Array.isArray(args) || (args.length < 3)) {
    throw new TypeError(`args is not an appropriate array.`)
  }

  // Get the onfiguration based on what was requested.
  // Note: args[3] may be undefined. This is ok and will be handled
  //       when the preset is Custom.
  const config = getConfiguration(args[2], args[3]);

  // Generate a single 'default' password
  try {
    const password = HSXKPasswd.passwordSync(config);
    // Make the password available to the calling KbdMaestro action.
    console.log(password);
  }
  catch(err) {
    console.log(err);
  }
}

/* ========================================================================
   Description:    Create a XSXKPasswd Configuration according to user request.

   @param {string} [argPreset]     - (Optional) Number indicating the password
                                                preset to use.
                                                Default:0 (Grumpy Default)
   @param {string} [argCustomData] - (Optional) JSON representation of the
                                                HSXKPasswd configuration to use.
                                                Default: Default config.

   @throws {TypeError}    - thrown when argPreset does not meet expectations.
   @throws {SyntaxError}  - thrown when argCustomData is not valid JSON.
======================================================================== */
function getConfiguration(argPreset, argCustomData) {

  const preset = Number.parseInt(argPreset, 10);
  if (isNaN(preset) || !(Object.values(XKPASSWD_PRESETS).includes(preset))) {
    throw TypeError(`Requested preset is invalid. val:'${argPreset}'`);
  }
  const customData = (typeof(argCustomData) !== 'undefined') ? JSON.parse(argCustomData) : undefined;

  // Get a default configuration and limits.
  const config = HSXKPasswd.Config.defaultSettings;

  // Adjust the configuration based on the desired preset.
  switch (preset) {
    case XKPASSWD_PRESETS.APPLEID:
    {
      config.case_transform             = 'RANDOM';
      config.word_length_min            = 5;
      config.word_length_max            = 7;
      config.separator_character        = 'RANDOM';
      config.separator_alphabet         = ['-', ':', '.', ','];
      config.padding_digits_before      = 1;
      config.padding_digits_after       = 1;
      config.padding_type               = 'FIXED';
      config.padding_characters_before  = 1;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '?', '@', '&'];
    }
    break;
    case XKPASSWD_PRESETS.SECURITYQ :
    {
      config.case_transform             = 'NONE';
      config.num_words                  = 6;
      config.word_length_min            = 4;
      config.word_length_max            = 8;
      config.separator_character        = ' ';
      config.padding_digits_before      = 0;
      config.padding_digits_after       = 0;
      config.padding_type               = 'FIXED';
      config.padding_characters_before  = 0;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '?', '!'];
    }
    break;
    case XKPASSWD_PRESETS.WEB16:
    {
      config.case_transform             = 'RANDOM';
      config.word_length_min            = 4;
      config.word_length_max            = 4;
      config.separator_character        = 'RANDOM';
      config.separator_alphabet         = ['-', '+', '=', '.', '*', '_', '|', '~', ','];
      config.padding_digits_before      = 0;
      config.padding_digits_after       = 0;
      config.padding_type               = 'FIXED';
      config.padding_characters_before  = 1;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '@', '$', '%', '^', '&', '*', '+', '=', ':', '|', '~', '?'];

    }
    break;
    case XKPASSWD_PRESETS.WEB32:
    {
      config.case_transform             = 'ALTERNATE';
      config.num_words                  = 4;
      config.word_length_min            = 4;
      config.word_length_max            = 5;
      config.separator_character        = 'RANDOM';
      config.separator_alphabet         = ['-', '+', '=', '.', '*', '_', '|', '~', ','];
      config.padding_digits_before      = 2;
      config.padding_digits_after       = 2;
      config.padding_type               = 'FIXED';
      config.padding_characters_before  = 1;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '@', '$', '%', '^', '&', '*', '+', '=', ':', '|', '~', '?'];
    }
    break;
    case XKPASSWD_PRESETS.WIFI:
    {
      config.case_transform             = 'RANDOM';
      config.num_words                  = 6;
      config.word_length_min            = 4;
      config.word_length_max            = 8;
      config.separator_character        = 'RANDOM';
      config.separator_alphabet         = ['-', '+', '=', '.', '*', '_', '|', '~', ','];
      config.padding_digits_before      = 4;
      config.padding_digits_after       = 4;
      config.padding_type               = 'ADAPTIVE';
      config.padding_characters_before  = 0;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '@', '$', '%', '^', '&', '*', '+', '=', ':', '|', '~', '?'];
      config.pad_to_length              = 63
    }
    break;
    case XKPASSWD_PRESETS.NTLM:
    {
      config.case_transform             = 'INVERT';
      config.num_words                  = 3;    // Should be 2. But implementation will not allow this to be <3. Bug#4
      config.word_length_min            = 5;
      config.word_length_max            = 5;
      config.separator_character        = 'RANDOM';
      config.separator_alphabet         = ['-', '+', '=', '.', '*', '_', '|', '~', ','];
      config.padding_digits_before      = 1;
      config.padding_digits_after       = 0;
      config.padding_type               = 'FIXED';
      config.padding_characters_before  = 0;
      config.padding_characters_after   = 1;
      config.padding_character          = 'RANDOM';
      config.padding_alphabet           = ['!', '@', '$', '%', '^', '&', '*', '+', '=', ':', '|', '~', '?'];
    }
    break;
    case XKPASSWD_PRESETS.XKCD:
    {
      config.case_transform             = 'RANDOM';
      config.num_words                  = 4;
      config.word_length_min            = 4;
      config.word_length_max            = 8;
      config.separator_character        = '-';
      config.padding_digits_before      = 0;
      config.padding_digits_after       = 0;
      config.padding_type               = 'NONE';
    }
    break;
    case XKPASSWD_PRESETS.CUSTOM:
    {
      // The Custom preset requires custom data.
      if (!customData) {
        throw new TypeError(`Custom Data is not valid.`);
      }

      // Get a default configuration and limits.
      const nullSettings = HSXKPasswd.Config.nullSettings;

      // Iterate over the items in the custom data
      // and update the configuration as appropriate.
      for (const [key, value] of Object.entries(customData)) {
        // Ensure this entry exists in the KSXKPwd config.
        if (config.hasOwnProperty(key) || nullSettings.hasOwnProperty(key)) {
          // Update the value for this item.
          switch (key)
          {
            // Integer cases
            case 'allow_accents':
            case 'num_words':
            case 'word_length_min':
            case 'word_length_max':
            case 'padding_characters_before':
            case 'padding_characters_after':
            case 'padding_digits_before':
            case 'padding_digits_after':
            case 'pad_to_length':
            {
              config[key] = Number.parseInt(value);
            }
            break;

            // Array cases
            case 'padding_alphabet':
            case 'separator_alphabet':
            case 'symbol_alphabet':
            {
              config[key] = Array.from(value);
            }
            break;

            case 'separator_character':
            case 'padding_character':
            {
              // If the value is something other than 'RANDOM'
              // then just use the first character...but allow unicode !!
              let tmpVal = value;
              if ((tmpVal !== 'NONE') && (tmpVal !== 'RANDOM'))
              {
                tmpVal = Array.from(value)[0];
              }
              config[key] = tmpVal;
            }
            break;

            // All others, treat as string.
            case 'case_transform':
            case 'padding_type':
            default:
            {
              config[key] = String(value);
            }
            break;
          }
        }
        else {
          throw new RangeError(`Custom config included unknown key: ${key}`);
        }
      }
    }
    break;

    case XKPASSWD_PRESETS.GRUMPY_DEF:
    {
      // Settings redacted for personal security considerations.
    }
    break;

    case XKPASSWD_PRESETS.DEFAULT:
    default:
    {
      // Leave the config in the default condition.
    }
    break;
  }

  // Convert the configuration data into a real configuration.
  const configuration = new HSXKPasswd.Config(config);

  return configuration;
}

/* ========================================================================
   Script implementation starts here !
======================================================================== */
main(process.argv);
